/*
 *  Project: Auto-Hide Sticky Header
 *  Description: This creates a header that will automaticly appear and disappear
 *  Author: Jasper Boeijenga
 *  License: MIT
 */

;(function ( $, window, undefined ) {

	// Create the defaults once
	var pluginName = 'stickyUpHeader',
		document = window.document,
		$window	= $( window ),
		defaults = {
			showClasses : 'visible-stuck-up visible-stuck-up-scroll',
			temporaryHideClasses : 'visible-stuck-up',
			hideClasses : 'visible-stuck-up-scroll',
			throttleTimeout	: 500,
		};

	// The actual plugin constructor
	function Plugin( element, options ) {
		this.$element = $(element);
		this.options = $.extend( {}, defaults, options) ;
		if(this.options.scrollHeight !== undefined){
			this.options.height = this.$element.height();
		}

		this._defaults = defaults;
		this._name = pluginName;
		this.init();
	}

	Plugin.prototype.init = function(){
		var position = 0;
		var _this = this;

		$window.on( 'scroll', _this.throttle( this.options.throttleTimeout, function(){
			var scrollTop = $window.scrollTop();
			if( scrollTop <= 0 ) {
				_this.hide();
			} else if( scrollTop < position ) {
				if( scrollTop > 400 ) {
					_this.show();
				} else {
					_this.hide();
				}
			} else {
				_this.hideTemp();
			}
			position = scrollTop;
		}));
	};


	Plugin.prototype.throttle = function( delay, fn )
	{
		var last, deferTimer;
		return function()
		{
			var context = this, args = arguments, now = +new Date;
			if( last && now < last + delay )
			{
				clearTimeout( deferTimer );
				deferTimer = setTimeout( function(){ last = now; fn.apply( context, args ); }, delay );
			}
			else
			{
				last = now;
				fn.apply( context, args );
			}
		};
	};

	Plugin.prototype.hide = function(){
		return this.$element.removeClass(this.options.hideClasses);
	};

	Plugin.prototype.hideTemp = function(){
		return this.$element.removeClass(this.options.temporaryHideClasses);
	};

	Plugin.prototype.show = function(){
		return this.$element.addClass(this.options.showClasses);
	};

	// A really lightweight plugin wrapper around the constructor, 
	// preventing against multiple instantiations
	$.fn[pluginName] = function ( options ) {
		return this.each(function () {
			if (!$.data(this, pluginName)) {
				$.data(this, pluginName, new Plugin( this, options ));
			}
		});
	};

}(jQuery, window));